// Zorg Subtitle - Premiere Pro Extension
// Main Application Logic

class ZorgSubtitleExtension {
  constructor() {
    this.csInterface = new CSInterface();
    this.apiKey = null;
    this.apiBaseUrl = 'https://zorgsubtitle.com/api/v1';
    this.currentVideoId = null;
    this.currentSequence = null;
    this.accountInfo = null;

    this.init();
  }

  init() {
    // Load saved API key
    this.loadApiKey();

    // Setup event listeners
    this.setupEventListeners();

    // If API key exists, auto-connect
    if (this.apiKey) {
      document.getElementById('api-key').value = this.apiKey;
      this.connect();
    }

    // Update theme based on Premiere Pro
    this.updateTheme();
  }

  setupEventListeners() {
    // Authentication
    document.getElementById('connect-btn').addEventListener('click', () => this.connect());
    document.getElementById('disconnect-btn').addEventListener('click', () => this.disconnect());
    document.getElementById('dashboard-link').addEventListener('click', (e) => {
      e.preventDefault();
      this.openDashboard();
    });

    // Sequence
    document.getElementById('get-active-sequence').addEventListener('click', () => this.getActiveSequence());

    // Export & Upload
    document.getElementById('export-upload-btn').addEventListener('click', () => this.exportAndUpload());

    // Transcription
    document.getElementById('transcribe-btn').addEventListener('click', () => this.transcribe());

    // Translation
    document.getElementById('translate-btn').addEventListener('click', () => this.translate());

    // Import
    document.getElementById('import-btn').addEventListener('click', () => this.importSubtitles());
  }

  // API Key Management
  loadApiKey() {
    this.apiKey = localStorage.getItem('zorg_api_key');
  }

  saveApiKey(key) {
    this.apiKey = key;
    localStorage.setItem('zorg_api_key', key);
  }

  clearApiKey() {
    this.apiKey = null;
    localStorage.removeItem('zorg_api_key');
  }

  // Authentication
  async connect() {
    const apiKeyInput = document.getElementById('api-key').value.trim();
    const statusEl = document.getElementById('auth-status');

    if (!apiKeyInput) {
      this.showStatus(statusEl, 'Please enter your API key', 'error');
      return;
    }

    // Validate API key format
    if (!apiKeyInput.startsWith('zs_')) {
      this.showStatus(statusEl, 'Invalid API key format. Should start with "zs_"', 'error');
      return;
    }

    this.showStatus(statusEl, 'Connecting...', 'info');

    try {
      // Test API key by fetching account info
      const response = await fetch(`${this.apiBaseUrl}/account`, {
        headers: {
          'Authorization': `Bearer ${apiKeyInput}`,
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Authentication failed');
      }

      this.accountInfo = await response.json();
      this.saveApiKey(apiKeyInput);

      this.showStatus(statusEl, 'Connected successfully!', 'success');

      // Show main content
      document.getElementById('auth-section').style.display = 'none';
      document.getElementById('main-content').style.display = 'block';

      // Update account info
      this.updateAccountInfo();

    } catch (error) {
      console.error('Connection error:', error);
      this.showStatus(statusEl, `Connection failed: ${error.message}`, 'error');
    }
  }

  disconnect() {
    this.clearApiKey();
    this.accountInfo = null;
    this.currentVideoId = null;
    this.currentSequence = null;

    document.getElementById('auth-section').style.display = 'block';
    document.getElementById('main-content').style.display = 'none';
    document.getElementById('api-key').value = '';

    // Reset UI
    this.resetUI();
  }

  updateAccountInfo() {
    if (!this.accountInfo) return;

    const creditsUsed = this.accountInfo.credits_used || 0;
    const creditsLimit = this.accountInfo.credits_limit || 0;
    const creditsRemaining = creditsLimit - creditsUsed;

    document.getElementById('credits-remaining').textContent =
      `${creditsRemaining} / ${creditsLimit}`;

    document.getElementById('user-plan').textContent =
      this.accountInfo.subscription_tier || 'Free';
  }

  // Premiere Pro Integration
  async getActiveSequence() {
    const infoBox = document.getElementById('sequence-info');

    try {
      const result = await this.evalScript('getActiveSequenceInfo()');
      const sequenceInfo = JSON.parse(result);

      if (!sequenceInfo.name) {
        throw new Error('No active sequence found');
      }

      this.currentSequence = sequenceInfo;

      infoBox.innerHTML = `
        <strong>${sequenceInfo.name}</strong><br>
        Duration: ${this.formatTime(sequenceInfo.duration)}<br>
        Frame Rate: ${sequenceInfo.frameRate} fps<br>
        Resolution: ${sequenceInfo.width}x${sequenceInfo.height}
      `;
      infoBox.classList.add('active');

      // Enable export button
      document.getElementById('export-upload-btn').disabled = false;

    } catch (error) {
      console.error('Error getting sequence:', error);
      infoBox.innerHTML = `<span style="color: var(--error);">Error: ${error.message}</span>`;
      infoBox.classList.add('active');
    }
  }

  async exportAndUpload() {
    if (!this.currentSequence) {
      alert('Please select a sequence first');
      return;
    }

    const quality = document.getElementById('export-quality').value;
    const progressContainer = document.getElementById('upload-progress');
    const progressBar = document.getElementById('upload-progress-bar');
    const statusText = document.getElementById('upload-status');

    progressContainer.style.display = 'block';
    document.getElementById('export-upload-btn').disabled = true;

    try {
      // Step 1: Export video from Premiere Pro
      statusText.textContent = 'Exporting video from Premiere Pro...';
      progressBar.style.width = '10%';

      const exportPath = await this.evalScript(`exportSequence("${quality}")`);
      const exportResult = JSON.parse(exportPath);

      if (!exportResult.path) {
        throw new Error('Export failed');
      }

      statusText.textContent = 'Reading exported file...';
      progressBar.style.width = '30%';

      // Step 2: Read file as base64
      const fileData = await this.readFileAsBase64(exportResult.path);

      statusText.textContent = 'Uploading to Zorg Subtitle...';
      progressBar.style.width = '50%';

      // Step 3: Upload to API
      const uploadResponse = await fetch(`${this.apiBaseUrl}/videos`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          name: this.currentSequence.name,
          file: fileData,
          filename: `${this.currentSequence.name}.mp4`
        })
      });

      if (!uploadResponse.ok) {
        const error = await uploadResponse.json();
        throw new Error(error.error || 'Upload failed');
      }

      const uploadResult = await uploadResponse.json();
      this.currentVideoId = uploadResult.id;

      progressBar.style.width = '100%';
      statusText.textContent = 'Upload complete! Video ID: ' + this.currentVideoId;

      // Enable transcribe button
      document.getElementById('transcribe-btn').disabled = false;

      // Clean up temp file
      await this.evalScript(`deleteFile("${exportResult.path}")`);

    } catch (error) {
      console.error('Export/Upload error:', error);
      statusText.textContent = 'Error: ' + error.message;
      statusText.style.color = 'var(--error)';
    } finally {
      document.getElementById('export-upload-btn').disabled = false;
    }
  }

  async transcribe() {
    if (!this.currentVideoId) {
      alert('Please upload a video first');
      return;
    }

    const language = document.getElementById('language').value;
    const statusEl = document.getElementById('transcribe-status');

    this.showStatus(statusEl, 'Starting transcription...', 'info');
    document.getElementById('transcribe-btn').disabled = true;

    try {
      const response = await fetch(`${this.apiBaseUrl}/videos/${this.currentVideoId}/transcribe`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          language: language === 'auto' ? null : language
        })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Transcription failed');
      }

      const result = await response.json();

      this.showStatus(statusEl, 'Transcription started! This may take a few minutes...', 'info');

      // Poll for completion
      await this.pollTranscriptionStatus(statusEl);

    } catch (error) {
      console.error('Transcription error:', error);
      this.showStatus(statusEl, `Error: ${error.message}`, 'error');
      document.getElementById('transcribe-btn').disabled = false;
    }
  }

  async pollTranscriptionStatus(statusEl) {
    const pollInterval = 5000; // 5 seconds
    let attempts = 0;
    const maxAttempts = 120; // 10 minutes max

    const poll = async () => {
      try {
        const response = await fetch(`${this.apiBaseUrl}/videos/${this.currentVideoId}`, {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`
          }
        });

        const video = await response.json();

        if (video.status === 'completed') {
          this.showStatus(statusEl, 'Transcription completed!', 'success');
          document.getElementById('translate-btn').disabled = false;
          document.getElementById('import-btn').disabled = false;
          return;
        } else if (video.status === 'failed') {
          throw new Error('Transcription failed');
        } else if (video.status === 'processing') {
          attempts++;
          if (attempts < maxAttempts) {
            this.showStatus(statusEl, `Processing... (${Math.round(attempts * pollInterval / 1000)}s)`, 'info');
            setTimeout(poll, pollInterval);
          } else {
            throw new Error('Transcription timeout');
          }
        }
      } catch (error) {
        this.showStatus(statusEl, `Error: ${error.message}`, 'error');
        document.getElementById('transcribe-btn').disabled = false;
      }
    };

    poll();
  }

  async translate() {
    if (!this.currentVideoId) {
      alert('Please transcribe a video first');
      return;
    }

    const targetLanguage = document.getElementById('translate-language').value;
    const statusEl = document.getElementById('translate-status');

    if (!targetLanguage) {
      this.showStatus(statusEl, 'Please select a target language', 'error');
      return;
    }

    this.showStatus(statusEl, 'Translating subtitles...', 'info');
    document.getElementById('translate-btn').disabled = true;

    try {
      const response = await fetch(`${this.apiBaseUrl}/videos/${this.currentVideoId}/translate`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          target_language: targetLanguage
        })
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Translation failed');
      }

      this.showStatus(statusEl, 'Translation completed!', 'success');
      document.getElementById('import-btn').disabled = false;

    } catch (error) {
      console.error('Translation error:', error);
      this.showStatus(statusEl, `Error: ${error.message}`, 'error');
    } finally {
      document.getElementById('translate-btn').disabled = false;
    }
  }

  async importSubtitles() {
    if (!this.currentVideoId) {
      alert('Please generate subtitles first');
      return;
    }

    const statusEl = document.getElementById('import-status');
    this.showStatus(statusEl, 'Fetching subtitles...', 'info');
    document.getElementById('import-btn').disabled = true;

    try {
      // Fetch subtitles
      const response = await fetch(`${this.apiBaseUrl}/videos/${this.currentVideoId}/subtitles`, {
        headers: {
          'Authorization': `Bearer ${this.apiKey}`
        }
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to fetch subtitles');
      }

      const subtitles = await response.json();

      if (!subtitles || subtitles.length === 0) {
        throw new Error('No subtitles found');
      }

      this.showStatus(statusEl, 'Importing to Premiere Pro...', 'info');

      // Import to Premiere Pro
      const subtitlesJson = JSON.stringify(subtitles);
      const result = await this.evalScript(`importSubtitles('${this.escapeString(subtitlesJson)}')`);

      this.showStatus(statusEl, 'Subtitles imported successfully!', 'success');

    } catch (error) {
      console.error('Import error:', error);
      this.showStatus(statusEl, `Error: ${error.message}`, 'error');
    } finally {
      document.getElementById('import-btn').disabled = false;
    }
  }

  // Utility Functions
  async evalScript(script) {
    return new Promise((resolve, reject) => {
      this.csInterface.evalScript(script, (result) => {
        if (result === 'EvalScript error.') {
          reject(new Error('Script execution failed'));
        } else {
          resolve(result);
        }
      });
    });
  }

  async readFileAsBase64(filePath) {
    // Use Node.js fs module (enabled via CEFCommandLine)
    const fs = require('fs');
    const fileBuffer = fs.readFileSync(filePath);
    return fileBuffer.toString('base64');
  }

  showStatus(element, message, type) {
    element.textContent = message;
    element.className = `status ${type}`;
  }

  formatTime(seconds) {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = Math.floor(seconds % 60);
    return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  }

  escapeString(str) {
    return str.replace(/'/g, "\\'").replace(/\n/g, '\\n').replace(/\r/g, '\\r');
  }

  updateTheme() {
    // Get Premiere Pro color scheme
    const bgColor = this.csInterface.getHostEnvironment().appSkinInfo.panelBackgroundColor.color;
    document.body.style.backgroundColor =
      `rgb(${bgColor.red}, ${bgColor.green}, ${bgColor.blue})`;
  }

  resetUI() {
    document.getElementById('sequence-info').innerHTML = '';
    document.getElementById('sequence-info').classList.remove('active');
    document.getElementById('upload-progress').style.display = 'none';
    document.getElementById('export-upload-btn').disabled = true;
    document.getElementById('transcribe-btn').disabled = true;
    document.getElementById('translate-btn').disabled = true;
    document.getElementById('import-btn').disabled = true;
  }

  openDashboard() {
    const url = 'https://zorgsubtitle.com/dashboard/api-keys';
    this.csInterface.openURLInDefaultBrowser(url);
  }
}

// Initialize extension when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  window.zorgExtension = new ZorgSubtitleExtension();
});
