// CSInterface.js - Adobe CEP Interface Library
// Simplified version for Premiere Pro extensions

var CSInterface = function() {};

// System paths
CSInterface.prototype.getSystemPath = function(pathType) {
  var path = decodeURI(window.__adobe_cep__.getSystemPath(pathType));
  return path;
};

// Extension path
CSInterface.prototype.getExtensionPath = function() {
  return this.getSystemPath('extension');
};

// Evaluate ExtendScript
CSInterface.prototype.evalScript = function(script, callback) {
  if (callback === null || callback === undefined) {
    callback = function(result) {};
  }
  window.__adobe_cep__.evalScript(script, callback);
};

// Get host environment
CSInterface.prototype.getHostEnvironment = function() {
  var hostEnv = window.__adobe_cep__.getHostEnvironment();
  return JSON.parse(hostEnv);
};

// Get host capabilities
CSInterface.prototype.getHostCapabilities = function() {
  var hostCapabilities = window.__adobe_cep__.getHostCapabilities();
  return JSON.parse(hostCapabilities);
};

// Dispatch event
CSInterface.prototype.dispatchEvent = function(event) {
  if (typeof event.data === 'object') {
    event.data = JSON.stringify(event.data);
  }
  window.__adobe_cep__.dispatchEvent(event);
};

// Add event listener
CSInterface.prototype.addEventListener = function(type, listener, obj) {
  window.__adobe_cep__.addEventListener(type, listener, obj);
};

// Remove event listener
CSInterface.prototype.removeEventListener = function(type, listener, obj) {
  window.__adobe_cep__.removeEventListener(type, listener, obj);
};

// Request to open a URL in the default browser
CSInterface.prototype.openURLInDefaultBrowser = function(url) {
  return window.cep.util.openURLInDefaultBrowser(url);
};

// Get scale factor
CSInterface.prototype.getScaleFactor = function() {
  var hostEnv = this.getHostEnvironment();
  return hostEnv.appSkinInfo.scaleFactor;
};

// Set window title
CSInterface.prototype.setWindowTitle = function(title) {
  window.__adobe_cep__.invokeSync('setWindowTitle', title);
};

// Get application ID
CSInterface.prototype.getApplicationID = function() {
  var hostEnv = this.getHostEnvironment();
  return hostEnv.appId;
};

// Close extension
CSInterface.prototype.closeExtension = function() {
  window.__adobe_cep__.closeExtension();
};

// Register invalid certificate callback
CSInterface.prototype.registerInvalidCertificateCallback = function(callback) {
  return window.__adobe_cep__.registerInvalidCertificateCallback(callback);
};

// Register keyevents callback
CSInterface.prototype.registerKeyEventsCallback = function(callback) {
  return window.__adobe_cep__.registerKeyEventsCallback(callback);
};

// Set context menu
CSInterface.prototype.setContextMenu = function(menu, callback) {
  window.__adobe_cep__.invokeAsync('setContextMenu', menu, callback);
};

// Set context menu by JSON
CSInterface.prototype.setContextMenuByJSON = function(menu, callback) {
  window.__adobe_cep__.invokeAsync('setContextMenuByJSON', menu, callback);
};

// Update context menu
CSInterface.prototype.updateContextMenuItem = function(menuItemID, enabled, checked) {
  var ret = window.__adobe_cep__.updateContextMenuItem(menuItemID, enabled, checked);
  return ret;
};

// Is window visible
CSInterface.prototype.isWindowVisible = function() {
  return window.__adobe_cep__.isWindowVisible();
};

// Resize window
CSInterface.prototype.resizeContent = function(width, height) {
  window.__adobe_cep__.resizeContent(width, height);
};

// CSEvent class
var CSEvent = function(type, scope, appId, extensionId) {
  this.type = type;
  this.scope = scope || 'GLOBAL';
  this.appId = appId;
  this.extensionId = extensionId;
  this.data = '';
};

// System path constants
CSInterface.SystemPath = {
  USER_DATA: 'userData',
  COMMON_FILES: 'commonFiles',
  MY_DOCUMENTS: 'myDocuments',
  APPLICATION: 'application',
  EXTENSION: 'extension',
  HOST_APPLICATION: 'hostApplication'
};

// Extension constants
CSInterface.EvalScript_ErrMessage = 'EvalScript error.';

// Color types
var ColorType = function() {};
ColorType.RGB = 'rgb';
ColorType.GRADIENT = 'gradient';
ColorType.NONE = 'none';

// UI color
var UIColor = function(red, green, blue, alpha) {
  this.color = {
    red: red || 0,
    green: green || 0,
    blue: blue || 0,
    alpha: alpha || 255
  };
  this.type = ColorType.RGB;
};

// Gradient stop
var GradientStop = function(offset, rgbColor) {
  this.offset = offset;
  this.rgbColor = rgbColor;
};

// Gradient color
var GradientColor = function(gradientStops) {
  this.type = ColorType.GRADIENT;
  this.gradientStops = gradientStops;
};

// App skin info
var AppSkinInfo = function(baseFontFamily, baseFontSize, appBarBackgroundColor,
                          panelBackgroundColor, appBarBackgroundColorSRGB,
                          panelBackgroundColorSRGB, systemHighlightColor) {
  this.baseFontFamily = baseFontFamily;
  this.baseFontSize = baseFontSize;
  this.appBarBackgroundColor = appBarBackgroundColor;
  this.panelBackgroundColor = panelBackgroundColor;
  this.appBarBackgroundColorSRGB = appBarBackgroundColorSRGB;
  this.panelBackgroundColorSRGB = panelBackgroundColorSRGB;
  this.systemHighlightColor = systemHighlightColor;
};

// Host environment
var HostEnvironment = function(appId, appVersion, appLocale, appUILocale, appSkinInfo) {
  this.appId = appId;
  this.appVersion = appVersion;
  this.appLocale = appLocale;
  this.appUILocale = appUILocale;
  this.appSkinInfo = appSkinInfo;
};

// Host capabilities
var HostCapabilities = function(EXTENDED_PANEL_MENU, EXTENDED_PANEL_ICONS, DELEGATE_APE_ENGINE,
                                SUPPORT_HTML_EXTENSIONS, DISABLE_FLASH_EXTENSIONS) {
  this.EXTENDED_PANEL_MENU = EXTENDED_PANEL_MENU;
  this.EXTENDED_PANEL_ICONS = EXTENDED_PANEL_ICONS;
  this.DELEGATE_APE_ENGINE = DELEGATE_APE_ENGINE;
  this.SUPPORT_HTML_EXTENSIONS = SUPPORT_HTML_EXTENSIONS;
  this.DISABLE_FLASH_EXTENSIONS = DISABLE_FLASH_EXTENSIONS;
};

// API version
var ApiVersion = function(major, minor, micro) {
  this.major = major || 0;
  this.minor = minor || 0;
  this.micro = micro || 0;
};

// Menu item status
var MenuItemStatus = function(menuItemLabel, enabled, checked) {
  this.menuItemLabel = menuItemLabel;
  this.enabled = enabled;
  this.checked = checked;
};

// Context menu item
var ContextMenuItem = function(id, label, enabled, checked) {
  this.id = id;
  this.label = label;
  this.enabled = enabled;
  this.checked = checked;
};
