// Zorg Subtitle - ExtendScript for Premiere Pro
// This script runs in Premiere Pro's ExtendScript engine

#include "json2.js"

// Get active sequence information
function getActiveSequenceInfo() {
  try {
    var project = app.project;
    if (!project) {
      return JSON.stringify({ error: 'No active project' });
    }

    var activeSequence = project.activeSequence;
    if (!activeSequence) {
      return JSON.stringify({ error: 'No active sequence' });
    }

    var frameRate = activeSequence.timebase;
    var duration = activeSequence.end / frameRate;

    var sequenceInfo = {
      name: activeSequence.name,
      duration: duration,
      frameRate: frameRate,
      width: activeSequence.frameSizeHorizontal,
      height: activeSequence.frameSizeVertical,
      videoTracks: activeSequence.videoTracks.numTracks,
      audioTracks: activeSequence.audioTracks.numTracks
    };

    return JSON.stringify(sequenceInfo);

  } catch (e) {
    return JSON.stringify({ error: e.toString() });
  }
}

// Export sequence to file
function exportSequence(quality) {
  try {
    var project = app.project;
    var activeSequence = project.activeSequence;

    if (!activeSequence) {
      return JSON.stringify({ error: 'No active sequence' });
    }

    // Create temp directory for export
    var tempFolder = Folder.temp.fsName + '/ZorgSubtitle/';
    var folder = new Folder(tempFolder);
    if (!folder.exists) {
      folder.create();
    }

    // Generate filename
    var timestamp = new Date().getTime();
    var filename = activeSequence.name.replace(/[^a-z0-9]/gi, '_') + '_' + timestamp + '.mp4';
    var outputPath = tempFolder + filename;

    // Set up encoder preset
    var encoder = app.encoder;
    var preset = getPresetPath(quality);

    if (!preset) {
      return JSON.stringify({ error: 'Export preset not found' });
    }

    // Add to encoder queue
    var outputFile = new File(outputPath);

    encoder.encodeSequence(
      activeSequence,
      outputFile.fsName,
      preset,
      0, // workAreaType - 0 for entire sequence
      0  // removeOnCompletion
    );

    // Start encoding
    encoder.startBatch();

    // Wait for encoding to complete
    var maxWaitTime = 300; // 5 minutes max
    var waited = 0;
    var checkInterval = 2; // 2 seconds

    while (encoder.isEncodingActive() && waited < maxWaitTime) {
      $.sleep(checkInterval * 1000);
      waited += checkInterval;
    }

    if (encoder.isEncodingActive()) {
      return JSON.stringify({ error: 'Export timeout' });
    }

    // Check if file exists
    if (!outputFile.exists) {
      return JSON.stringify({ error: 'Export file not created' });
    }

    return JSON.stringify({
      path: outputFile.fsName,
      size: outputFile.length
    });

  } catch (e) {
    return JSON.stringify({ error: e.toString() });
  }
}

// Get export preset path based on quality
function getPresetPath(quality) {
  // Premiere Pro preset paths
  var presetFolder = app.path + '/MediaIO/systempresets/4D4F4F56_4D505347/HD/';

  var presetName = quality === 'high'
    ? 'HD 1080p 29.97.epr'
    : 'HD 720p 29.97.epr';

  var presetFile = new File(presetFolder + presetName);

  if (presetFile.exists) {
    return presetFile.fsName;
  }

  // Fallback: try to find any H.264 preset
  var systemPresets = new Folder(app.path + '/MediaIO/systempresets/');
  var h264Folder = new Folder(systemPresets.fsName + '/4D4F4F56_4D505347/');

  if (h264Folder.exists) {
    var files = h264Folder.getFiles('*.epr');
    if (files.length > 0) {
      return files[0].fsName;
    }
  }

  return null;
}

// Import subtitles to timeline
function importSubtitles(subtitlesJson) {
  try {
    var subtitles = JSON.parse(subtitlesJson);
    var project = app.project;
    var activeSequence = project.activeSequence;

    if (!activeSequence) {
      return JSON.stringify({ error: 'No active sequence' });
    }

    // Create a graphics layer for subtitles
    var videoTrackIndex = activeSequence.videoTracks.numTracks - 1;

    if (videoTrackIndex < 0) {
      // Add a video track if none exist
      activeSequence.videoTracks.addTrack();
      videoTrackIndex = 0;
    }

    var videoTrack = activeSequence.videoTracks[videoTrackIndex];

    // Create a temp SRT file
    var tempFolder = Folder.temp.fsName + '/ZorgSubtitle/';
    var folder = new Folder(tempFolder);
    if (!folder.exists) {
      folder.create();
    }

    var srtFilePath = tempFolder + 'subtitles_' + new Date().getTime() + '.srt';
    var srtFile = new File(srtFilePath);
    srtFile.encoding = 'UTF-8';
    srtFile.open('w');

    // Convert subtitles to SRT format
    for (var i = 0; i < subtitles.length; i++) {
      var sub = subtitles[i];
      srtFile.writeln((i + 1).toString());
      srtFile.writeln(formatSRTTime(sub.start) + ' --> ' + formatSRTTime(sub.end));
      srtFile.writeln(sub.text);
      srtFile.writeln('');
    }

    srtFile.close();

    // Import SRT file
    var imported = project.importFiles([srtFilePath]);

    if (imported) {
      // Find the imported item
      var projectItems = project.rootItem.children;
      var srtItem = null;

      for (var j = 0; j < projectItems.numItems; j++) {
        var item = projectItems[j];
        if (item.name.indexOf('.srt') !== -1) {
          srtItem = item;
          break;
        }
      }

      if (srtItem) {
        // Add to timeline
        var frameRate = activeSequence.timebase;
        videoTrack.insertClip(srtItem, 0);

        return JSON.stringify({
          success: true,
          count: subtitles.length,
          message: 'Subtitles imported successfully'
        });
      }
    }

    return JSON.stringify({ error: 'Failed to import SRT file' });

  } catch (e) {
    return JSON.stringify({ error: e.toString() });
  }
}

// Format time for SRT (HH:MM:SS,mmm)
function formatSRTTime(seconds) {
  var hours = Math.floor(seconds / 3600);
  var minutes = Math.floor((seconds % 3600) / 60);
  var secs = Math.floor(seconds % 60);
  var millis = Math.floor((seconds % 1) * 1000);

  return pad(hours, 2) + ':' + pad(minutes, 2) + ':' + pad(secs, 2) + ',' + pad(millis, 3);
}

function pad(num, size) {
  var s = num.toString();
  while (s.length < size) s = '0' + s;
  return s;
}

// Delete temporary file
function deleteFile(filePath) {
  try {
    var file = new File(filePath);
    if (file.exists) {
      file.remove();
      return 'true';
    }
    return 'false';
  } catch (e) {
    return 'false';
  }
}

// Utility function to log messages
function log(message) {
  $.writeln('[Zorg Subtitle] ' + message);
}
